#include "html.h"
#include "box.h"
#include "html_tag.h"
#include <document.h>

// TODO: Find more generic trace switch for litehtml
//#define ENABLE_LITEHTML_DEBUG_TRACE

litehtml::block_box::~block_box()
{
    if (m_element && (m_element->m_box == this)) {
        m_element->m_box = 0;
    }
}

litehtml::box_type litehtml::block_box::get_type()
{
    return box_block;
}

int litehtml::block_box::height()
{
    return m_element->height();
}

int litehtml::block_box::width()
{
    return m_element->width();
}

void litehtml::block_box::add_element(const element::ptr &el)
{
    m_element = el;
    el->m_box = this;
}

void litehtml::block_box::finish(int& /*ret_width*/, bool last_box)
{
    if(!m_element) return;
    m_element->apply_relative_shift(m_box_right - m_box_left);
}

bool litehtml::block_box::can_hold(const element::ptr &el, white_space ws)
{
    if(m_element || el->is_inline_box())
    {
        return false;
    }
    return true;
}

bool litehtml::block_box::is_empty()
{
    if(m_element)
    {
        return false;
    }
    return true;
}

int litehtml::block_box::baseline()
{
    if(m_element)
    {
        return m_element->get_base_line();
    }
    return 0;
}

void litehtml::block_box::get_elements( elements_vector& els )
{
    els.push_back(m_element);
}

int litehtml::block_box::top_margin()
{
    if(m_element && m_element->collapse_top_margin())
    {
        return m_element->m_margins.top;
    }
    return 0;
}

int litehtml::block_box::bottom_margin()
{
    if(m_element && m_element->collapse_bottom_margin())
    {
        return m_element->m_margins.bottom;
    }
    return 0;
}

void litehtml::block_box::y_shift( int shift )
{
    m_box_top += shift;
    if(m_element)
    {
        m_element->m_pos.y += shift;
    }
}

void litehtml::block_box::new_width( int left, int right, elements_vector& els )
{

}

//////////////////////////////////////////////////////////////////////////


litehtml::line_box::~line_box()
{
    elements_vector::const_iterator it(m_items.begin());
    if (it != m_items.end())
    {
        if ((*it)->m_box == this) {
            (*it)->m_box = 0;
        }
    }
}

litehtml::box_type litehtml::line_box::get_type()
{
    return box_line;
}

int litehtml::line_box::height()
{
    return m_height;
}

int litehtml::line_box::width()
{
    return m_width;
}

void litehtml::line_box::add_element(const element::ptr &el)
{
    el->m_skip  = false;
    el->m_box   = 0;
    bool add    = true;
    if( (m_items.empty() && el->is_white_space()) || el->is_break() )
    {
        el->m_skip = true;
    } else if(el->is_white_space())
    {
        if (have_last_space())
        {
            add = false;
            el->m_skip = true;
        }
    }

    if(add)
    {
        el->m_box = this;
        m_items.push_back(el);

        if(!el->m_skip)
        {
            int el_shift_left   = el->get_inline_shift_left();
            int el_shift_right  = el->get_inline_shift_right();

            el->m_pos.x = m_box_left + m_width + el_shift_left + el->content_margins_left();
            el->m_pos.y = m_box_top + el->content_margins_top();

            position rect;
            el->m_doc->container()->get_client_rect(rect);

            int width = el->width() + el_shift_left + el_shift_right;
            m_width += width;

        }
    }
}

void litehtml::line_box::finish(int& ret_width, bool last_box)
{
    bool isEmpty = is_empty();
    if( isEmpty || (!isEmpty && last_box && is_break_only()) )
    {
        m_height = 0;
        return;
    }

    bool useBaselineOffset = false;
    int baselineOffset = -1;
    elements_vector::const_iterator it(m_items.begin());
    if (it != m_items.end())
    {
        // if a baseline offset is defined for the document, it only affects the first line!
        baselineOffset = (*it)->get_document()->GetBaselineOffest();
        useBaselineOffset = (baselineOffset >= 0) && (m_lineIndex == 0);
    }

    for(elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend(); i++)
    {
        if((*i)->is_white_space() || (*i)->is_break())
        {
            if(!(*i)->m_skip)
            {
                (*i)->m_skip = true;
                m_width -= (*i)->width();
            }
        }
        else
        {
            break;
        }
    }

    LayoutDirection baseLayoutDir = Undefined;
    LayoutDirection lastElLayoutDir = Neutral;
    int line_width = (m_box_right - m_box_left);
    int left_border = 0;
    int right_border = line_width;
    int blockWidth = -1;
    int blockOffset = -1;

    if (line_width > 1) {
        // e.g. for table cells line_width is set to 1 to check minimum size
        // for that use case ignore truncation handling
        ProcessTruncation(ret_width, line_width);
    }

    document* doc = 0;
    if (m_items.begin() != m_items.end()) {
        const element::ptr& firstChunk = (*m_items.begin());
        doc = firstChunk->get_document();
        baseLayoutDir = firstChunk->GetBaseLayoutDirection();
        if (baseLayoutDir == Neutral) {
            baseLayoutDir = firstChunk->GetLayoutDirection();
            element::ptr parent = firstChunk->parent();
            // in case of RTL:
            // If the layout direction is taken from first chunk, it has to be provided to all next lines.
            // Storing it to the parent solves the problem.
            if (baseLayoutDir == RightToLeft && parent.IsValid()) {
                parent->SetBaseLayoutDirection(baseLayoutDir);
            }
        }
    }

    bool cultureDependentAlignment = true;
    bool rightToLeftCulture = false;
    if (doc != 0) {
        const document_container::ptr& container = doc->container();
        if (container.IsValid()) {
            cultureDependentAlignment = container->IsCultureDependentAlignment();
            rightToLeftCulture = container->IsRightToLeftCulture();
        }
    }

    text_align textAlign = m_text_align;
    if ((!rightToLeftCulture && (baseLayoutDir == RightToLeft)) ||
        (cultureDependentAlignment && rightToLeftCulture && (baseLayoutDir != RightToLeft)) ||
        (!cultureDependentAlignment && rightToLeftCulture && (baseLayoutDir == RightToLeft))) {
        if (textAlign == text_align_left) {
            textAlign = text_align_right;
        }
        else if (textAlign == text_align_right) {
            textAlign = text_align_left;
        }
    }

    int base_line = m_font_metrics.base_line();
    int line_height = 0;

    int add_x = 0;
    switch (textAlign)
    {
        case text_align_right:
            add_x = (m_box_right - m_box_left) - m_width;
            break;
        case text_align_center:
            add_x = ((m_box_right - m_box_left) - m_width) / 2;
            break;
        default:
            add_x = 0;
    }

    m_height = 0;
    if (useBaselineOffset)
    {
        line_height = baselineOffset + m_font_metrics.descent;
    } else {
        if(m_height)
        {
            base_line += (line_height - m_height) / 2;
        }
    }

    // find line box baseline and line-height
    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        const element::ptr& el = (*it);

        if (el->get_display() == display_inline_text)
        {
            font_metrics fm;
            el->get_font(&fm);
            base_line = MAX(base_line, fm.base_line());
            if (!useBaselineOffset) {
                line_height = MAX(line_height, el->line_height());
            }
            m_height = MAX(m_height, fm.height);
        }
        else if (el->get_display() == display_inline_block && is_break_only())
        {
            if (!useBaselineOffset) {
                line_height = MAX(line_height, el->line_height());
            }
        }
        el->m_pos.x += add_x;

#ifdef ENABLE_LITEHTML_DEBUG_TRACE
        litehtml::tstring elText;
        litehtml::el_text* el_text = dynamic_cast<litehtml::el_text*>(&(*el));
        if (el_text != 0)
        {
            el_text->get_text(elText);
        }
        LayoutDirection lytDir = el->GetLayoutDirection();
        printf("->> <%s>: %i %s / %s '%s'\n", el->get_tagName(), el->m_pos.x,
        (Undefined == baseLayoutDir ? "!Undef!" : (Neutral == baseLayoutDir ? "Neutral" : (RightToLeft == baseLayoutDir ? "<-- RtL" : "LtR -->"))),
        (Undefined == lytDir ? "!Undef!" : (Neutral == lytDir ? "Neutral" : (RightToLeft == lytDir ? "<-- RtL" : "LtR -->"))),
        (el_text != 0 ? elText.c_str() : "-X-"));
#endif
    }

    // process horizontal, BiDi capable text box layout
    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        const element::ptr& el = (*it);

        LayoutDirection layoutDir = el->GetLayoutDirection();
        if (el->IsLayoutDirNeutral()) {
            layoutDir = (Neutral == lastElLayoutDir ? baseLayoutDir : lastElLayoutDir);
        } else {
            lastElLayoutDir = layoutDir;
        }

        if (RightToLeft == baseLayoutDir) {
            // BiDi text in right to left context

            if (RightToLeft == layoutDir)
            {
                // mirror the elements of a line horizontally according to their direction.
                // mirror RtL elements within the given boundaries
                el->m_pos.x = (right_border - (el->m_pos.x + el->width()));
                blockWidth = -1;
            }
            else
            {
                // process all LtR elements in this RtL context and place them
                // to the left of the last RtL block (or at the right area border):
                // shift_x = (mirred_x1 - x1) - total_width
                // where mirrored_x1 is X pos of the first element mirrored as above
                //       and x1 is the regular LtR X pos of the first element
                //       and total_width is the width of all elements inc. spaces.
                // and for each of these elements: m_pos.x += shift_x

                // process all subsequent LtR (and neutral) boxes as one block
                // 1. determine the total width of that LtR block
                if (blockWidth < 0) {
                    blockWidth = 0;
                    blockOffset = 0;
                    for (elements_vector::iterator itLtr = it; itLtr != m_items.end() && ((*itLtr)->GetLayoutDirection() != RightToLeft); ++itLtr)
                    {
                        blockWidth += (*itLtr)->width();
                    }
                    // 2. calculate the distance for translating all elements of the LtR block once
                    blockOffset = right_border - (2 * el->m_pos.x + blockWidth);
                }
                // 3. move all elements of that LtR block
                el->m_pos.x += blockOffset;
            }
        }
        else
        {
            // BiDi text in left to right context
            if (RightToLeft == layoutDir)
            {
                // process all subsequent RtL (and neutral) boxes as one block
                // 1. determine the total width of that RtL block
                if (blockOffset < 0) {
                    blockOffset = el->m_pos.x;
                    for (elements_vector::iterator itRtl = it; itRtl != m_items.end() && ((*itRtl)->GetLayoutDirection() != LeftToRight); ++itRtl)
                    {
                       right_border = (*itRtl)->m_pos.x + (*itRtl)->width();
                    }
                 }
                 // 2. mirror the elements of block horizontally within the block boundaries
                 el->m_pos.x = ((blockOffset + right_border) - (el->m_pos.x + el->width()));
             } else {
             blockOffset = -1;
             right_border = line_width;
          }
       }
    }

    m_height = line_height;

    int y1  = 0;
    int y2  = m_height;

    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        if(el->get_display() == display_inline_text)
        {
            font_metrics fm;
            el->get_font(&fm);
            el->m_pos.y = m_height - base_line - fm.ascent;
        } else
        {
            switch(el->get_vertical_align())
            {
            case va_super:
            case va_sub:
            case va_baseline:
                el->m_pos.y = m_height - base_line - el->height() + el->get_base_line() + el->content_margins_top();
                break;
            case va_top:
                el->m_pos.y = y1 + el->content_margins_top();
                break;
            case va_text_top:
                el->m_pos.y = m_height - base_line - m_font_metrics.ascent + el->content_margins_top();
                break;
            case va_middle:
                el->m_pos.y = m_height - base_line - m_font_metrics.x_height / 2 - el->height() / 2 + el->content_margins_top();
                break;
            case va_bottom:
                el->m_pos.y = y2 - el->height() + el->content_margins_top();
                break;
            case va_text_bottom:
                el->m_pos.y = m_height - base_line + m_font_metrics.descent - el->height() + el->content_margins_top();
                break;
            }
            y1 = MIN(y1, el->top());
            y2 = MAX(y2, el->bottom());
        }
    }

    css_offsets offsets;

    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        if (!useBaselineOffset)
        {
            el->m_pos.y -= y1;
            el->m_pos.y += m_box_top;
        }
        if(el->get_display() != display_inline_text)
        {
            switch(el->get_vertical_align())
            {
            case va_top:
                el->m_pos.y = m_box_top + el->content_margins_top();
                break;
            case va_bottom:
                el->m_pos.y = m_box_top + (y2 - y1) - el->height() + el->content_margins_top();
                break;
            case va_baseline:
                //TODO: process vertical align "baseline"
                break;
            case va_middle:
                //TODO: process vertical align "middle"
                break;
            case va_sub:
                //TODO: process vertical align "sub"
                break;
            case va_super:
                //TODO: process vertical align "super"
                break;
            case va_text_bottom:
                //TODO: process vertical align "text-bottom"
                break;
            case va_text_top:
                //TODO: process vertical align "text-top"
                break;
            }
        }

        el->apply_relative_shift(m_box_right - m_box_left);
    }
    if (!useBaselineOffset)
    {
        m_height = y2 - y1;
    }
    m_baseline = (base_line - y1) - (m_height - line_height);
}

void litehtml::line_box::ProcessTruncation(int& ret_width, int line_width)
{
    // truncation: detect line and text-element to truncate
    element::ptr& el = (*m_items.begin());
    document* doc = el->get_document();
    const OverflowPropValue* overflowAttrib = dynamic_cast<const OverflowPropValue*>(el->GetStyleProperty(PropValue::Overflow, true));
    const TextOverflowPropValue* textOverflowAttrib = dynamic_cast<const TextOverflowPropValue*>(el->GetStyleProperty(PropValue::TextOverflow, true));

    // process truncation only if element is leaf and the last line is too long
    bool processTruncation = (overflowAttrib != 0) && (overflowAttrib->m_value == litehtml::overflow_hidden)
        && (textOverflowAttrib != 0)
        && ((line_width >= 0) && (m_width > line_width))
        && ((doc->GetMaxNumberOfLines() == 0) || ((m_lineIndex + 1) >= doc->GetMaxNumberOfLines()))
        && (el->get_children_count() == 0);

    if (processTruncation) {
        litehtml::text_overflow textOverflowType = textOverflowAttrib->m_type;
        litehtml::tstring truncationText = textOverflowAttrib->m_value;
        int truncationTextWidth = 0;
        int truncationWidth = -1;
        el_text::ptr elTruncationText;

        if (textOverflowType == text_overflow_clip || textOverflowType == text_overflow_string) {
            if (textOverflowType == text_overflow_string && !truncationText.empty()) {
                // document element for truncation text is only created and configured if such a text is present
                elTruncationText = el_text::ptr(LITEHTML_NEW(el_text)(truncationText.c_str(), doc));
                if (elTruncationText.IsValid()) {
                    el->appendChild(elTruncationText);
                    elTruncationText->parse_styles();
                }

                // apply size of truncation text according to display: inline-text rules.
                // note: html_tag->place_element(...) does not work here for this purpose.
                size truncationTextSize;
                elTruncationText->get_content_size(truncationTextSize, line_width);
                elTruncationText->m_pos = truncationTextSize;
                truncationTextWidth = elTruncationText->m_pos.width;
            }

            elements_vector::iterator truncationItemPos = m_items.end();
            for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it) {
                const element::ptr& el = (*it);
                int itemWidth = el->m_pos.x + el->m_pos.width;
                if (((el->m_pos.x + truncationTextWidth) < line_width) && ((itemWidth + truncationTextWidth) >= line_width)) {
                    if (truncationWidth < 0) {
                        litehtml::el_text* el_text = dynamic_cast<litehtml::el_text*>(&(*el));
                        if (el_text != 0) {
                            truncationWidth = el_text->GetGlyphAlignedWidth(line_width - el->m_pos.x - truncationTextWidth);
                        }
                        // clip element box to truncation width
                        el->m_pos.width = truncationWidth;
                        el->m_truncationWidth = truncationWidth;
                        truncationItemPos = it;
                        m_width = el->m_pos.x + el->m_pos.width + truncationTextWidth;
                        ret_width = line_width;

                        if (elTruncationText.IsValid()) {
                            elTruncationText->m_pos.x = el->m_pos.x + truncationWidth;
                            elTruncationText->m_pos.y = el->m_pos.y;
                        }
                    }
                }
                else if ((el->m_pos.x + truncationTextWidth) >= line_width) {
                    el->m_pos.x = line_width;
                    el->m_pos.width = 0;
                    el->m_truncationWidth = 0;
                }
            }
            if (elTruncationText.IsValid() && truncationItemPos != m_items.end()) {
                truncationItemPos++;
                m_items.insert(truncationItemPos, elTruncationText);
            }
        }
    }
}

bool litehtml::line_box::can_hold(const element::ptr &el, white_space ws)
{
    if (!el->is_inline_box())
    {
        return false;
    }

    const int maxNumberOfLines = el->get_document()->GetMaxNumberOfLines();
    const bool isTruncationLine = (maxNumberOfLines > 0) && (GetLineIndex() >= maxNumberOfLines - 1);

    if (!m_items.empty() && !isTruncationLine && m_items.back()->is_break())
    {
        return false;
    }

    if (isTruncationLine || el->is_break())
    {
        return true;
    }

    if (ws == white_space_nowrap || ws == white_space_pre)
    {
        return true;
    }

    if (m_box_left + m_width + el->width() + el->get_inline_shift_left() + el->get_inline_shift_right() > m_box_right)
    {
        return false;
    }

    return true;
}

bool litehtml::line_box::have_last_space()
{
    bool ret = false;
    for (elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend() && !ret; i++)
    {
        if((*i)->is_white_space() || (*i)->is_break())
        {
            ret = true;
        } else
        {
            break;
        }
    }
    return ret;
}

bool litehtml::line_box::is_empty()
{
    if(m_items.empty()) return true;
    for (elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend(); i++)
    {
        if(!(*i)->m_skip || (*i)->is_break())
        {
            return false;
        }
    }
    return true;
}

int litehtml::line_box::baseline()
{
    return m_baseline;
}

void litehtml::line_box::get_elements( elements_vector& els )
{
    els.insert(els.begin(), m_items.begin(), m_items.end());
}

int litehtml::line_box::top_margin()
{
    return 0;
}

int litehtml::line_box::bottom_margin()
{
    return 0;
}

void litehtml::line_box::y_shift( int shift )
{
    m_box_top += shift;
    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        el->m_pos.y += shift;
    }
}

bool litehtml::line_box::is_break_only()
{
    if(m_items.empty()) return true;

    if(m_items.front()->is_break())
    {
        for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
        {
            element::ptr& el = (*it);
            if(!el->m_skip)
            {
                return false;
            }
        }
        return true;
    }
    return false;
}

void litehtml::line_box::new_width(int left, int right, elements_vector& els)
{
    int add = left - m_box_left;
    if (add) {
        m_box_left = left;
        m_box_right = right;
        m_width = 0;
        elements_vector::iterator remove_begin = m_items.end();
        for (elements_vector::iterator i = m_items.begin() + 1; i != m_items.end(); i++) {
            element::ptr el = (*i);

            if (!el->m_skip) {
                if (m_box_left + m_width + el->width() + el->get_inline_shift_right() + el->get_inline_shift_left() > m_box_right) {
                    remove_begin = i;
                    break;
                }
                else {
                    el->m_pos.x += add;
                    m_width += el->width() + el->get_inline_shift_right() + el->get_inline_shift_left();
                }
            }
        }
        if (remove_begin != m_items.end()) {
            els.insert(els.begin(), remove_begin, m_items.end());
            m_items.erase(remove_begin, m_items.end());

            for (elements_vector::iterator it = els.begin(); it != els.end(); ++it) {
                element::ptr& el = (*it);
                el->m_box = 0;
            }
        }
    }
}

bool litehtml::line_box::IsLastLine(const element& elem) const
{
    int maxNumberOfLines = elem.get_document()->GetMaxNumberOfLines();

    return ((maxNumberOfLines > 0) && (GetLineIndex() == maxNumberOfLines - 1));
}

